#!/usr/bin/env python3
# -*- coding: utf-8 -*-
#
# ThoughtPy
# Version 1.0
# Copyright © 2017 Mark Damon Hughes
#
# Redistribution and use in source and binary forms, with or without
# modification, are permitted provided that the following conditions are
# met:
#
# 1. Redistributions of source code must retain the above copyright
# notice, this list of conditions and the following disclaimer.
#
# 2. Redistributions in binary form must reproduce the above copyright
# notice, this list of conditions and the following disclaimer in the
# documentation and/or other materials provided with the distribution.
#
# 3. Neither the name of the copyright holder nor the names of its
# contributors may be used to endorse or promote products derived from
# this software without specific prior written permission.
#
# THIS SOFTWARE IS PROVIDED BY THE COPYRIGHT HOLDERS AND CONTRIBUTORS "AS
# IS" AND ANY EXPRESS OR IMPLIED WARRANTIES, INCLUDING, BUT NOT LIMITED
# TO, THE IMPLIED WARRANTIES OF MERCHANTABILITY AND FITNESS FOR A
# PARTICULAR PURPOSE ARE DISCLAIMED. IN NO EVENT SHALL THE COPYRIGHT
# HOLDER OR CONTRIBUTORS BE LIABLE FOR ANY DIRECT, INDIRECT, INCIDENTAL,
# SPECIAL, EXEMPLARY, OR CONSEQUENTIAL DAMAGES (INCLUDING, BUT NOT LIMITED
# TO, PROCUREMENT OF SUBSTITUTE GOODS OR SERVICES; LOSS OF USE, DATA, OR
# PROFITS; OR BUSINESS INTERRUPTION) HOWEVER CAUSED AND ON ANY THEORY OF
# LIABILITY, WHETHER IN CONTRACT, STRICT LIABILITY, OR TORT (INCLUDING
# NEGLIGENCE OR OTHERWISE) ARISING IN ANY WAY OUT OF THE USE OF THIS
# SOFTWARE, EVEN IF ADVISED OF THE POSSIBILITY OF SUCH DAMAGE.

import os, sys, time, subprocess

USAGE = """ThoughtPy 1.0 by Mark Damon Hughes http://markdamonhughes.com/thoughtpy/
Usage: thought.py [OPTIONS]
-y YEAR		Edit YEAR, default = current year.
-t TOPIC	Edit subtopic TOPIC, default = main.
-l		List timestamp and first line of all notes in current file.
-f TEXT		List all entries which contain TEXT in current file. Must be last option.
-p		Print filename, which can then be passed to a markdown previewer.
-		Read TEXT from stdin instead of calling editor.

Environment vars:
THOUGHT_DIR	Directory to save notes, default ~/Dropbox/Apps/ThoughtPy/
THOUGHT_EDITOR	Editor to use, default is $EDITOR, or nano (a sample ~/.nanorc config file is on the web site above).
"""

def yearForTimestamp(t):
	"""
	>>> yearForTimestamp(0)
	'1969'
	"""
	return time.strftime("%Y", time.localtime(t))

def formatDateTime(t):
	"""
	>>> formatDateTime(0)
	'1969-12-31 16:00:00 PST -0800'
	"""
	return time.strftime("%Y-%m-%d %H:%M:%S %Z %z", time.localtime(t))

class ThoughtPy:
	def __init__(self):
		if "THOUGHT_DIR" in os.environ:
			self.notedir = os.path.expanduser(os.environ["THOUGHT_DIR"])
		else:
			self.notedir = os.path.expanduser("~/Dropbox/Apps/ThoughtPy/")

		if "THOUGHT_EDITOR" in os.environ:
			self.editor = os.environ["THOUGHT_EDITOR"]
		elif "EDITOR" in os.environ:
			self.editor = os.environ["EDITOR"]
		else:
			self.editor = "nano"

		self.timestamp = time.time()
		self.year = yearForTimestamp(self.timestamp)
		self.topic = "main"
		self.action = "edit"
		self.actionArgs = None

	def parseArgs(self, args):
		argc = len(args)
		i = 0
		while i < argc:
			a = args[i]
			i += 1
			if a == "-":
				self.action = "stdin"
			elif a == "-f" and i < argc:
				self.action = "find"
				self.actionArgs = args[i:]
				i = argc
			elif a == "-l":
				self.action = "list"
			elif a == "-p":
				self.action = "print"
			elif a == "-t" and i < argc:
				self.topic = args[i]
				i += 1
			elif a == "-y" and i < argc:
				self.year = int(args[i])
				i += 1
			elif a == "-h":
				print(USAGE)
				sys.exit(0)
			else:
				import doctest
				doctest.testmod()
				raise Exception("Unknown argument", a)

	def run(self):
		if self.action == "edit":
			self.doEdit()
		elif self.action == "list":
			self.doList()
		elif self.action == "find":
			self.doFind()
		elif self.action == "print":
			self.doPrint()
		elif self.action == "stdin":
			self.doStdin()
		else:
			raise Exception("Unknown action", self.action)

	def entry(self):
		"""Creates an entry and returns filename."""
		f = self.filename()
		self.makeDir(f)
		newfile = not os.path.exists(f)
		with open(f, "a") as fp:
			if newfile:
				fp.write("# %s %s\n\n" % (self.year, self.topic))
			else:
				fp.write("\n----\n")
			fp.write("## %s\n" % (formatDateTime(self.timestamp), ))
		return f

	def filename(self):
		return "%s/%s-%s.md" % (self.notedir, self.year, self.topic)

	def makeDir(self, f):
		os.makedirs(os.path.dirname(f), exist_ok=True)

	def parse(self):
		"""Parses file, returns {"year":str, "topic":str, "items":[str...]}
		"""
		data = {}
		data["items"] = []
		item = ""

		f = self.filename()
		with open(f, "r") as fp:
			while True:
				ln = fp.readline()
				if ln == "":
					if item: data["items"].append(item)
					break
				ln = ln.rstrip()
				if ln == "":
					continue
				if ln.startswith("# "):
					words = ln.split(maxsplit=2)
					data["year"] = words[1]
					data["topic"] = words[2]
				elif ln.startswith("## "):
					if item: data["items"].append(item)
					item = ln[3:]+"\n"
				elif ln == "----":
					pass
				else:
					item += ln+"\n"
		return data

	def doEdit(self):
		f = self.entry()
		cmdline = self.editor.split()
		cmdline.append("+32767") # force editor to last line
		cmdline.append(f)
		subprocess.run(cmdline)

	def doFind(self):
		f = self.filename()
		if not os.path.exists(f):
			print("File not found:", f)
			sys.exit(1)
		data = self.parse()
		print("#", data.get("year"), data.get("topic"))
		for it in data["items"]:
			found = True
			for a in self.actionArgs:
				if it.lower().find(a.lower()) < 0:
					found = False
					break
			if found:
				print("##", it)

	def doList(self):
		f = self.filename()
		if not os.path.exists(f):
			print("File not found:", f)
			sys.exit(1)
		data = self.parse()
		print("#", data.get("year"), data.get("topic"))
		for it in data["items"]:
			lns = it.splitlines()
			print("##", "\n".join(lns[0:2]))

	def doStdin(self):
		text = sys.stdin.read()
		f = self.entry()
		with open(f, "a") as fp:
			fp.write(text)

	def doPrint(self):
		print(self.filename())

if __name__ == '__main__':
	try:
		n = ThoughtPy()
		n.parseArgs(sys.argv[1:])
		n.run()
	except Exception as e:
		print(USAGE)
		raise e
